<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Quotes_model extends CRM_Model
{
    private $shipping_fields = array('shipping_street', 'shipping_city', 'shipping_city', 'shipping_state', 'shipping_zip', 'shipping_country');

    function __construct()
    {
        parent::__construct();
        $this->load->model('emails_model');
    }

    /**
     * Get quote by id
     * @param  mixed $id
     * @return array
     */
    public function get($id = '')
    {
        $this->db->select('*,tblcurrencies.id as currencyid, tblquotes.id as id, tblcurrencies.name as currency_name');
        $this->db->from('tblquotes');
        $this->db->join('tblcurrencies', 'tblcurrencies.id = tblquotes.currency', 'left');

        if (is_numeric($id)) {
            $this->db->where('tblquotes.id', $id);
            $quote = $this->db->get()->row();

            if ($quote) {
                $this->load->model('clients_model');
                $quote->items  = $this->get_quote_items($id);
                $quote->client = $this->clients_model->get($quote->clientid);
                if ($quote->client->company == '') {
                    $quote->client->company = $quote->client->firstname . ' ' . $quote->client->lastname;
                }
            }
            return $quote;
        }

        return $this->db->get()->result_array();
    }

    /**
     * Convert quote to invoice
     * @param  mixed $id quote id
     * @return mixed     New invoice ID
     */
    public function convert_to_invoice($id, $client = false)
    {
        // Recurring invoice date is okey lets convert it to new invoice
        $_quote = $this->get($id);

        $new_invoice_data             = array();
        $new_invoice_data['clientid'] = $_quote->clientid;
        $new_invoice_data['_number']  = get_option('next_invoice_number');
        $new_invoice_data['date']     = _d(date('Y-m-d'));
        $new_invoice_data['duedate']  = _d(date('Y-m-d'));


        if (get_option('invoice_due_after') != 0) {
            $new_invoice_data['duedate'] = _d(date('Y-m-d', strtotime('+' . get_option('invoice_due_after') . ' DAY', strtotime(date('Y-m-d')))));
        }

        $new_invoice_data['currency']         = $_quote->currency;
        $new_invoice_data['subtotal']         = $_quote->subtotal;
        $new_invoice_data['total']            = $_quote->total;
        $new_invoice_data['adjustment']       = $_quote->adjustment;
        $new_invoice_data['discount_percent'] = $_quote->discount_percent;
        $new_invoice_data['discount_total']   = $_quote->discount_total;
        $new_invoice_data['discount_type']    = $_quote->discount_type;
        $new_invoice_data['sale_agent']       = $_quote->sale_agent;

        // Since version 1.0.6
        $new_invoice_data['billing_street']           = $_quote->billing_street;
        $new_invoice_data['billing_city']             = $_quote->billing_city;
        $new_invoice_data['billing_state']            = $_quote->billing_state;
        $new_invoice_data['billing_zip']              = $_quote->billing_zip;
        $new_invoice_data['billing_country']          = $_quote->billing_country;
        $new_invoice_data['shipping_street']          = $_quote->shipping_street;
        $new_invoice_data['shipping_city']            = $_quote->shipping_city;
        $new_invoice_data['shipping_state']           = $_quote->shipping_state;
        $new_invoice_data['shipping_zip']             = $_quote->shipping_zip;
        $new_invoice_data['shipping_country']         = $_quote->shipping_country;
           if($_quote->include_shipping == 1){
             $new_invoice_data['include_shipping']         = 1;
           }
        $new_invoice_data['show_shipping_on_invoice'] = $_quote->show_shipping_on_quote;

        $new_invoice_data['terms']      = get_option('predefined_terms_invoice');
        $new_invoice_data['clientnote'] = get_option('predefined_clientnote_invoice');

        // Set to unpaid status automatically
        $new_invoice_data['status']     = 1;
        $new_invoice_data['clientnote'] = '';
        $new_invoice_data['adminnote']  = 'Converted from quote #' . format_quote_number($_quote->id);
        $this->load->model('payment_modes_model');

        $modes      = $this->payment_modes_model->get();
        $temp_modes = array();

        foreach ($modes as $mode) {
            $temp_modes[] = $mode['id'];
        }

        $new_invoice_data['allowed_payment_modes'] = $temp_modes;
        $new_invoice_data['newitems']              = array();

        $key = 1;
        foreach ($_quote->items as $item) {
            $new_invoice_data['newitems'][$key]['description']      = $item['description'];
            $new_invoice_data['newitems'][$key]['long_description'] = $item['long_description'];
            $new_invoice_data['newitems'][$key]['qty']              = $item['qty'];
            $new_invoice_data['newitems'][$key]['taxid']            = $item['taxid'];
            $new_invoice_data['newitems'][$key]['rate']             = $item['rate'];
            $new_invoice_data['newitems'][$key]['order']            = $item['item_order'];
            $key++;
        }

        $this->load->model('invoices_model');
        $id = $this->invoices_model->add($new_invoice_data);

        if ($id) {
            // Update quote with the new invoice data and set to status accepted
            $this->db->where('id', $_quote->id);
            $this->db->update('tblquotes', array(
                'invoiced_date' => date('Y-m-d H:i:s'),
                'invoiceid' => $id,
                'status' => 4
            ));
            if ($client == false) {
                $this->log_quote_activity($_quote->id, 'converted this quote to invoice.<br /><a href="' . admin_url('invoices/list_invoices/' . $id) . '">' . format_invoice_number($id) . '</a>');
            }
        }

        return $id;
    }

    public function get_quotes_total($data)
    {

        $statuses = array(
            1,
            2,
            3,
            4,
            5
        );
        $this->load->model('currencies_model');

        if ((is_using_multiple_currencies('tblquotes') && !isset($data['currency'])) || !isset($data['currency'])) {
            $currencyid = $this->currencies_model->get_base_currency()->id;
        } else if (isset($data['currency'])) {
            $currencyid = $data['currency'];
        }

        $symbol = $this->currencies_model->get_currency_symbol($currencyid);
        $sql    = 'SELECT';
        foreach ($statuses as $quote_status) {
            $sql .= '(SELECT SUM(total) FROM tblquotes WHERE status=' . $quote_status;
            $sql .= ' AND currency =' . $currencyid;
            $sql .= ') as "' . $quote_status . '",';
        }

        $sql     = substr($sql, 0, -1);
        $result  = $this->db->query($sql)->result_array();
        $_result = array();
        $i       = 0;
        foreach ($result as $key => $val) {
            foreach ($val as $total) {
                $_result[$i]['total']  = $total;
                $_result[$i]['symbol'] = $symbol;
                $i++;
            }
        }

        return $_result;
    }

    /**
     * Get all quote items
     * @param  mixed $id quoteid
     * @return array
     */

    public function get_quote_items($id)
    {
        $this->db->select('tblquoteitems.id,qty,rate,taxrate,tbltaxes.id as taxid,tbltaxes.name as taxname,description as description,long_description,item_order');
        $this->db->from('tblquoteitems');
        $this->db->join('tbltaxes', 'tbltaxes.id = tblquoteitems.taxid', 'left');
        $this->db->where('tblquoteitems.quoteid', $id);
        $this->db->order_by('item_order', 'asc');
        return $this->db->get()->result_array();
    }

    /**
     * Insert new quote to database
     * @param array $data invoiec data
     * @return mixed - false if not insert, quote ID if succes
     */
    public function add($data)
    {


        $data['number'] = $data['_number'];

        $unsetters = array(
            '_number',
            'currency_symbol',
            'price',
            'taxname',
            'description',
            'long_description',
            'taxid',
            'rate',
            'quantity',
            'item_select'
        );

        foreach ($unsetters as $unseter) {
            if (isset($data[$unseter])) {
                unset($data[$unseter]);
            }
        }

        if (isset($data['custom_fields'])) {
            $custom_fields = $data['custom_fields'];
            unset($data['custom_fields']);
        }

        $data['date'] = to_sql_date($data['date']);
        if (!empty($data['expirydate'])) {
            $data['expirydate'] = to_sql_date($data['expirydate']);
        } else {
            unset($data['expirydate']);
        }

        $data['hash'] = md5(rand() . microtime());
        // Check if the key exists
        $this->db->where('hash', $data['hash']);
        $exists = $this->db->get('tblquotes')->row();

        if ($exists) {
            $data['hash'] = md5(rand() . microtime());
        }

        $data['terms']       = nl2br($data['terms']);
        $data['datecreated'] = date('Y-m-d H:i:s');
        $data['year']        = get_option('quote_year');
        $data['addedfrom']   = get_staff_user_id();


        $items = array();
        if (isset($data['newitems'])) {
            $items = $data['newitems'];
            unset($data['newitems']);
        }

        if (!isset($data['include_shipping'])) {
            foreach ($this->shipping_fields as $_s_field) {
                if (isset($data[$_s_field])) {
                    $data[$_s_field] = NULL;
                }
            }
            $data['show_shipping_on_quote'] = 1;
            $data['include_shipping']          = 0;
        } else {
            $data['include_shipping'] = 1;
            // set by default for the next time to be checked
            if (isset($data['show_shipping_on_quote'])) {
                $data['show_shipping_on_quote'] = 1;
            } else {
                $data['show_shipping_on_quote'] = 0;
            }
        }

         if($data['discount_total'] == 0){
            $data['discount_type'] = '';
        }

        $_data = do_action('before_quote_added', array(
            'data' => $data,
            'items' => $items
        ));

        $data  = $_data['data'];
        $items = $_data['items'];

        $this->db->insert('tblquotes', $data);
        $insert_id = $this->db->insert_id();

        if ($insert_id) {

            if (isset($custom_fields)) {
                handle_custom_fields_post($insert_id, $custom_fields);
            }

            // Update next quote number in settings
            $this->db->where('name', 'next_quote_number');
            $this->db->set('value', 'value+1', FALSE);
            $this->db->update('tbloptions');
            if (count($items) > 0) {
                foreach ($items as $key => $item) {
                    $this->db->insert('tblquoteitems', array(
                        'description' => $item['description'],
                        'long_description' => nl2br($item['long_description']),
                        'qty' => $item['qty'],
                        'rate' => $item['rate'],
                        'taxid' => $item['taxid'],
                        'quoteid' => $insert_id,
                        'item_order' => $item['order']
                    ));
                }
            }

            $this->log_quote_activity($insert_id, 'created the quote');
            do_action('after_quote_added', $insert_id);
            return $insert_id;
        }

        return false;
    }

    public function get_quote_item($id)
    {
        $this->db->where('id', $id);
        return $this->db->get('tblquoteitems')->row();
    }

    /**
     * Update quote data
     * @param  array $data quote data
     * @param  mixed $id   quoteid
     * @return boolean
     */
    public function update($data, $id)
    {
        $affectedRows = 0;
        $prefix       = get_option('quote_prefix');

        $data['number'] = substr($data['number'], strlen($prefix));
        $data['number'] = trim($data['number']);

        if (get_option('quote_number_format') == 2) {
            $_temp_number   = explode('/', $data['number']);
            $data['number'] = $_temp_number[1];
        }

        $original_quote = $this->get($id);
        $original_status   = $original_quote->status;
        $original_number   = $original_quote->number;

        unset($data['currency_symbol']);
        unset($data['price']);
        unset($data['taxname']);
        unset($data['taxid']);
        unset($data['isedit']);
        unset($data['description']);
        unset($data['long_description']);
        unset($data['tax']);
        unset($data['rate']);
        unset($data['quantity']);
        unset($data['item_select']);


        $items = array();
        if (isset($data['items'])) {
            $items = $data['items'];
            unset($data['items']);
        }

        $newitems = array();
        if (isset($data['newitems'])) {
            $newitems = $data['newitems'];
            unset($data['newitems']);
        }

        if ($data['adjustment'] == 'NaN') {
            $data['adjustment'] = 0;
        }

        $data['terms']      = nl2br($data['terms']);
        $data['date']       = to_sql_date($data['date']);
        $data['expirydate'] = to_sql_date($data['expirydate']);


         if($data['discount_total'] == 0){
            $data['discount_type'] = '';
        }

        if (isset($data['custom_fields'])) {
            $custom_fields = $data['custom_fields'];
            if (handle_custom_fields_post($id, $custom_fields)) {
                $affectedRows++;
            }

            unset($data['custom_fields']);
        }

        if (!isset($data['include_shipping'])) {
            foreach ($this->shipping_fields as $_s_field) {
                if (isset($data[$_s_field])) {
                    $data[$_s_field] = NULL;
                }
            }
            $data['show_shipping_on_quote'] = 1;
            $data['include_shipping']          = 0;
        } else {
            $data['include_shipping'] = 1;
            // set by default for the next time to be checked
            if (isset($data['show_shipping_on_quote'])) {
                $data['show_shipping_on_quote'] = 1;
            } else {
                $data['show_shipping_on_quote'] = 0;
            }
        }

        $action_data = array(
            'data' => $data,
            'newitems' => $newitems,
            'items' => $items,
            'id' => $id,
            'removed_items' => array()
        );

        if (isset($data['removed_items'])) {
            $action_data['removed_items'] = $data['removed_items'];
        }

        $_data                 = do_action('before_quote_updated', $action_data);
        $data['removed_items'] = $_data['removed_items'];
        $newitems              = $_data['newitems'];
        $items                 = $_data['items'];
        $data                  = $_data['data'];

        // Delete items checked to be removed from database
        if (isset($data['removed_items'])) {
            foreach ($data['removed_items'] as $remove_item_id) {
                foreach ($data['removed_items'] as $remove_item_id) {
                    $original_item = $this->get_quote_item($remove_item_id);
                    $this->db->where('quoteid', $id);
                    $this->db->where('id', $remove_item_id);
                    $this->db->delete('tblquoteitems');
                    if ($this->db->affected_rows() > 0) {
                        $this->log_quote_activity($id, 'removed item <b>' . $original_item->description . '</b>');
                        $affectedRows++;
                    }
                }
            }
            unset($data['removed_items']);
        }

        $this->db->where('id', $id);
        $this->db->update('tblquotes', $data);

        if ($this->db->affected_rows() > 0) {
            // Check for status change
            if ($original_status != $data['status']) {
                $this->log_quote_activity($original_quote->id, 'Quote Status Updated: From: ' . format_quote_status($original_status) . ' To: ' . format_quote_status($data['status']));
            }

            if ($original_number != $data['number']) {
                $this->log_quote_activity($original_quote->id, 'Quote Number Changed: From: ' . format_quote_number($original_quote->id) . ' To: ' . format_quote_number($id));
            }

            $affectedRows++;

        }
        $this->load->model('taxes_model');
        if (count($items) > 0) {

            foreach ($items as $key => $item) {
                $quote_item_id = $item['itemid'];

                $original_item = $this->get_quote_item($quote_item_id);

                $this->db->where('id', $quote_item_id);
                $this->db->update('tblquoteitems', array(
                    'item_order' => $item['order']
                ));

                if ($this->db->affected_rows() > 0) {
                    $affectedRows++;
                }
                // Check for invoice item short description change
                $this->db->where('id', $quote_item_id);
                $this->db->update('tblquoteitems', array(
                    'description' => $item['description']
                ));

                if ($this->db->affected_rows() > 0) {
                    $this->log_quote_activity($id, 'updated item short description from ' . $original_item->description . ' to ' . $item['description']);
                    $affectedRows++;
                }

                // Check for item long description change
                $this->db->where('id', $quote_item_id);
                $this->db->update('tblquoteitems', array(
                    'long_description' => nl2br($item['long_description'])
                ));

                if ($this->db->affected_rows() > 0) {
                    $this->log_quote_activity($id, 'updated item long description from ' . $original_item->long_description . ' to ' . $item['long_description']);
                    $affectedRows++;
                }
                // Check for tax id change
                $original_tax = $this->taxes_model->get($original_item->taxid);

                $this->db->where('id', $quote_item_id);
                $this->db->update('tblquoteitems', array(
                    'taxid' => $item['taxid']
                ));

                if ($this->db->affected_rows() > 0) {
                    $tax_now = $this->taxes_model->get($item['taxid']);
                    if ($original_tax) {
                        if ($tax_now) {
                            $this->log_quote_activity($id, 'updated tax (' . $original_tax->name . ') from ' . $original_tax->taxrate . '% to (' . $tax_now->name . ') ' . $tax_now->taxrate . '%');
                        } else {
                            $this->log_quote_activity($id, 'removed tax (' . $original_tax->name . ') ' . $original_tax->taxrate . '%');
                        }
                    } else {
                        $this->log_quote_activity($id, 'added tax (' . $tax_now->name . ') ' . $tax_now->taxrate . '%');
                    }
                    $affectedRows++;
                }

                // Check for item rate change
                $this->db->where('id', $quote_item_id);
                $this->db->update('tblquoteitems', array(
                    'rate' => $item['rate']
                ));
                if ($this->db->affected_rows() > 0) {
                    $this->log_quote_activity($id, 'updated item rate from ' . $original_item->rate . ' to ' . $item['rate']);
                    $affectedRows++;
                }
                // CHeck for invoice quantity change
                $this->db->where('id', $quote_item_id);
                $this->db->update('tblquoteitems', array(
                    'qty' => $item['qty']
                ));

                if ($this->db->affected_rows() > 0) {
                    $this->log_quote_activity($id, 'updated quantity on item <b>' . $item['description'] . '</b> from ' . $original_item->qty . ' to ' . $item['qty']);
                    $affectedRows++;
                }
            }
        }

        if (count($newitems) > 0) {
            foreach ($newitems as $key => $item) {
                $this->db->insert('tblquoteitems', array(
                    'description' => $item['description'],
                    'long_description' => nl2br($item['long_description']),
                    'qty' => $item['qty'],
                    'rate' => $item['rate'],
                    'taxid' => $item['taxid'],
                    'quoteid' => $id,
                    'item_order' => $item['order']
                ));

                $new_item_added = $this->db->insert_id();
                if ($new_item_added) {
                    $this->log_quote_activity($id, 'added new item <b>' . $item['description'] . '</b>');
                    $affectedRows++;
                }
            }
        }

        if ($affectedRows > 0) {
            do_action('after_quote_updated', $id);
            return true;
        }

        return false;
    }

    public function mark_action_status($action, $id, $client = false)
    {
        $this->db->where('id', $id);
        $this->db->update('tblquotes', array(
            'status' => $action
        ));

        if ($this->db->affected_rows() > 0) {

            $quote  = $this->get($id);

            if($client == true){
                $this->db->where('staffid',$quote->addedfrom);
                $this->db->or_where('staffid',$quote->sale_agent);
                $staff_quote = $this->db->get('tblstaff')->result_array();

                $invoiceid = false;
                $invoiced  = false;
                if ($action == 4) {
                    if (get_option('quote_auto_convert_to_invoice_on_client_accept') == 1) {
                        $invoiceid = $this->convert_to_invoice($id, true);
                        $this->load->model('invoices_model');
                        if ($invoiceid) {
                            $invoiced = true;
                            $invoice  = $this->invoices_model->get($invoiceid);
                            $this->log_quote_activity($id, 'Client accepted this quote. Quote is converted to invoice with number <a href="' . admin_url('invoices/list_invoices/' . $invoiceid) . '">' . format_invoice_number($invoice->id) . '</a>', false, true);
                        }
                    } else {
                        $this->log_quote_activity($id, 'Client accepted this quote.', false, true);
                    }

                    // Send thank you email
                    $this->emails_model->send_email_template('quote-thank-you-to-customer', $quote->client->email, $quote->clientid, false, false, false, $id);

                    // add notifications for all users which have permissions manageSales
                    foreach ($staff_quote as $member) {
                        if (has_permission('manageSales', $member['staffid'])) {
                            add_notification(array(
                                'fromcompany' => true,
                                'touserid' => $member['staffid'],
                                'description' => 'Congratiolations! Client accepted quote with number ' . format_quote_number($quote->id),
                                'link' => 'quotes/list_quotes/' . $id
                            ));
                          // Send staff email notification that customer accepted quote
                          $this->emails_model->send_email_template('quote-accepted-to-staff', $member['email'], $quote->clientid, false, false, false, $id);
                        }
                    }

                    return array(
                        'invoiced' => $invoiced,
                        'invoiceid' => $invoiceid
                    );

                } else if ($action == 3) {
                    // add notifications for all users which have permissions manageSales
                    foreach ($staff_quote as $member) {
                        if (has_permission('manageSales', $member['staffid'])) {

                            add_notification(array(
                                'fromcompany' => true,
                                'touserid' => $member['staffid'],
                                'description' => 'Client declined quote with number ' . format_quote_number($quote->id),
                                'link' => 'quotes/list_quotes/' . $id
                            ));

                            // Send staff email notification that customer declined quote
                            $this->emails_model->send_email_template('quote-declined-to-staff', $member['email'], $quote->clientid, false, false, false, $id);

                        }
                    }
                    $this->log_quote_activity($id, 'Client declined this quote.', false, true);
                    return array(
                        'invoiced' => $invoiced,
                        'invoiceid' => $invoiceid
                    );
                }
            } else {
                // Admin marked quote
                $this->log_quote_activity($id,'marked quote as '.format_quote_status($status,'',false));
                return true;
            }
        }
        return false;
    }
    /**
     * Delete quote items and all connections
     * @param  mixed $id quoteid
     * @return boolean
     */
    public function delete($id)
    {

        if (get_option('delete_only_on_last_quote') == 1) {
            if (!is_last_quote($id)) {
                return false;
            }
        }


       $quote = $this->get($id);
       if(!is_null($quote->invoiceid)){
             return array('is_invoiced_quote_delete_error'=>true);
       }
        do_action('before_quote_deleted', $id);
        $this->db->where('id', $id);
        $this->db->delete('tblquotes');

        if ($this->db->affected_rows() > 0) {
            if (get_option('quote_number_decrement_on_delete') == 1) {
                $current_next_quote_number = get_option('next_quote_number');
                if ($current_next_quote_number > 1) {
                    // Decrement next quote number to
                    $this->db->where('name', 'next_quote_number');
                    $this->db->set('value', 'value-1', FALSE);
                    $this->db->update('tbloptions');
                }
            }
            $this->db->where('quoteid', $id);
            $this->db->delete('tblquoteitems');

            $this->db->where('quoteid', $id);
            $this->db->delete('tblquoteactivity');

            // Delete the custom field values
            $this->db->where('relid', $id);
            $this->db->where('fieldto', 'invoice');
            $this->db->delete('tblcustomfieldsvalues');

            return true;
        }
        return false;
    }
    /**
     * Set quote to sent when email is successfuly sended to client
     * @param mixed $id quoteid
     */
    public function set_quote_sent($id)
    {
        $this->db->where('id', $id);
        $this->db->update('tblquotes', array(
            'sent' => 1,
            'datesend' => date('Y-m-d H:i:s')
        ));

        $description = 'sent quote to client';
        $this->log_quote_activity($id, $description);

        // Update quote status to sent
        $this->db->where('id', $id);
        $this->db->update('tblquotes', array(
            'status' => 2
        ));
    }

    /**
     * Sent quote to client
     * @param  mixed  $id        quoteid
     * @param  string  $template  email template to sent
     * @param  boolean $attachpdf attach quote pdf or not
     * @return boolean
     */
    public function sent_quote_to_client($id, $template = '', $attachpdf = true)
    {

        $this->load->model('emails_model');
        $quote = $this->get($id);
        if ($template == '') {
            if ($quote->sent == 0) {
                $template = 'quote-send-to-client';
            } else {
                $template = 'quote-already-send';
            }
        }

        $quote_number = format_quote_number($quote->id);
        $pdf             = quote_pdf($quote);

        if ($attachpdf) {
            $attach = $pdf->Output($quote_number . '.pdf', 'S');
            $this->emails_model->add_attachment(array(
                'attachment' => $attach,
                'filename' => $quote_number . '.pdf',
                'type' => 'application/pdf'
            ));
        }

        $send = $this->emails_model->send_email_template($template, $quote->client->email, $quote->clientid, false, false, false, $id);
        if ($send) {
            $this->set_quote_sent($id);
            return true;
        }

        return false;
    }
    /**
     * All quote activity
     * @param  mixed $id quoteid
     * @return array
     */
    public function get_quote_activity($id)
    {
        $this->db->where('quoteid', $id);
        $this->db->order_by('date', 'asc');
        return $this->db->get('tblquoteactivity')->result_array();
    }
    /**
     * Log quote activity to database
     * @param  mixed $id   quoteid
     * @param  string $description activity description
     */
    public function log_quote_activity($id, $description = '', $cron = false, $client = false)
    {
        $staffid = get_staff_user_id();

        if ($cron == true) {
            $staffid = 'Cron Job';
        } else if ($client == true) {
            $staffid = NULL;
        }

        $this->db->insert('tblquoteactivity', array(
            'description' => $description,
            'date' => date('Y-m-d H:i:s'),
            'quoteid' => $id,
            'staffid' => $staffid
        ));
    }
}
